//34410A_34411A VISA and VCNET 2005 Sampled Measurements Example Program - This program 
// sets the 34410A or 34411A for 10 DCV measurements on the 10V range with a 1 second 
// delay between the measuremnts. The measurements are started with a BUS trigger (i.e., 
// software trigger). The program also checks the instrument id to make sure there is
// communication between the computer and instrument.
//
// For GPIB, the program uses the status byte to determine when measurements are made; 
// for USB and LAN, the program reads the Standard Event Register
//
// Included is an error checking routine to make sure the SCPI commands executed have
// the correct syntax.
//
// The program requires that the VISA drivers are installed in the computer. VISA 
// comes with the Agilent I/O Library. Also add the visa32.lib file as a resource and 
// include the visa.h file in the project.
//
// The program can use either GPIB, USB, or LAN to control the instrument. For GPIB, 
// use the appropriate VISA interface name (e.g., GPIB0, GPIB1, etc) and GPIB address
// (e.g., 9, 10, etc). For USB, use the correct interface name (e.g., USB0, USB1,
// etc) and ID string. Since the manufacturer's ID and instrument model code is
// already entered into the program, only the instrument serial number is required
// to run the program. For LAN, use the correct interface name (e.g., TCPIP0, TCPIP1,
// etc) and IP address of the instrument.
//
// The program was developed in Microsoft Visual C++ .NET 2005 as a WIN32 project.
//
//////////////////////////////////////////////////////////////////////////////////////
// Copyright  2006 Agilent Technologies Inc. All rights reserved.
//
// You have a royalty-free right to use, modify, reproduce and distribute this
// example files (and/or any modified version) in any way you find useful, provided
// that you agree that Agilent has no warranty, obligations or liability for any
// Sample Application Files.
//
// Agilent Technologies will not modify the program to provide added functionality or 
// construct procedures to meet your specific needs.
//////////////////////////////////////////////////////////////////////////////////////

#pragma once

#using <mscorlib.dll>

#include <visa.h>  // Required for VISA operation

// Declare pointers and variables
bool		connected = false;	// Used to determine instrument connection
ViSession	vi;             // Session identifier of devices
ViSession	videfaultRM;	// Resource manager session returned by viOpenDefaultRM(videfaultRM)
ViStatus	errorStatus;    // VISA function status return code
int			numrdgs;		// Used for the number of readings taken
double		tottime;		// Used to calculate total measurement time


namespace VISAExample
{
	using namespace System;
	using namespace System::ComponentModel;
	using namespace System::Collections;
	using namespace System::Windows::Forms;
	using namespace System::Data;
	using namespace System::Drawing;
	using namespace System::Runtime::InteropServices;

	/// <summary> 
	/// Summary for VISAExampleForm
	///
	/// WARNING: If you change the name of this class, you will need to change the 
	///          'Resource File Name' property for the managed resource compiler tool 
	///          associated with all .resx files this class depends on.  Otherwise,
	///          the designers will not be able to interact properly with localized
	///          resources associated with this form.
	/// </summary>
	public __gc class VISAExampleForm : public System::Windows::Forms::Form
	{	
	public:
		VISAExampleForm(void)
		{
			InitializeComponent();
		}
  
	protected:
		void Dispose(Boolean disposing)
		{
			if (disposing && components)
			{
				components->Dispose();
			}
			__super::Dispose(disposing);
		}
	public private: System::Windows::Forms::Label *  Label1;
	private: System::Windows::Forms::GroupBox *  SelectIOType;
	private: System::Windows::Forms::RadioButton *  LANType;
	private: System::Windows::Forms::RadioButton *  USBType;
	private: System::Windows::Forms::RadioButton *  GPIBType;
	private: System::Windows::Forms::TextBox *  lanaddr;
	private: System::Windows::Forms::TextBox *  usbaddr;
	private: System::Windows::Forms::TextBox *  gpibaddr;
	private: System::Windows::Forms::Label *  label4;
	private: System::Windows::Forms::Label *  label3;
	private: System::Windows::Forms::Label *  label2;

	private: System::Windows::Forms::ListBox *  listBox1;
	private: System::Windows::Forms::Button *  GetReadings;
	private: System::Windows::Forms::Button *  ExitProg;


	private:
		/// <summary>
		/// Required designer variable.
		/// </summary>
		System::ComponentModel::Container * components;

		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		void InitializeComponent(void)
		{
			this->Label1 = (new System::Windows::Forms::Label());
			this->SelectIOType = (new System::Windows::Forms::GroupBox());
			this->LANType = (new System::Windows::Forms::RadioButton());
			this->USBType = (new System::Windows::Forms::RadioButton());
			this->GPIBType = (new System::Windows::Forms::RadioButton());
			this->lanaddr = (new System::Windows::Forms::TextBox());
			this->usbaddr = (new System::Windows::Forms::TextBox());
			this->gpibaddr = (new System::Windows::Forms::TextBox());
			this->label4 = (new System::Windows::Forms::Label());
			this->label3 = (new System::Windows::Forms::Label());
			this->label2 = (new System::Windows::Forms::Label());
			this->listBox1 = (new System::Windows::Forms::ListBox());
			this->GetReadings = (new System::Windows::Forms::Button());
			this->ExitProg = (new System::Windows::Forms::Button());
			this->SelectIOType->SuspendLayout();
			this->SuspendLayout();
			// 
			// Label1
			// 
			this->Label1->Location = System::Drawing::Point(0, 8);
			this->Label1->Name = S"Label1";
			this->Label1->Size = System::Drawing::Size(408, 16);
			this->Label1->TabIndex = 81;
			this->Label1->Text = S"Select the I/O type and enter the appropriate data; then click on \"Get Readings\"";
			this->Label1->TextAlign = System::Drawing::ContentAlignment::TopCenter;
			// 
			// SelectIOType
			// 
			this->SelectIOType->Controls->Add(this->LANType);
			this->SelectIOType->Controls->Add(this->USBType);
			this->SelectIOType->Controls->Add(this->GPIBType);
			this->SelectIOType->Font = (new System::Drawing::Font(S"Microsoft Sans Serif", 8.25F, System::Drawing::FontStyle::Regular, System::Drawing::GraphicsUnit::Point, 
				(System::Byte)0));
			this->SelectIOType->Location = System::Drawing::Point(6, 24);
			this->SelectIOType->Name = S"SelectIOType";
			this->SelectIOType->Size = System::Drawing::Size(104, 88);
			this->SelectIOType->TabIndex = 82;
			this->SelectIOType->TabStop = false;
			this->SelectIOType->Text = S"Select I/O Type";
			// 
			// LANType
			// 
			this->LANType->Location = System::Drawing::Point(20, 64);
			this->LANType->Name = S"LANType";
			this->LANType->Size = System::Drawing::Size(57, 17);
			this->LANType->TabIndex = 4;
			this->LANType->Text = S"LAN";
			this->LANType->CheckedChanged += new System::EventHandler(this, &VISAExampleForm::LANType_CheckedChanged_1);
			// 
			// USBType
			// 
			this->USBType->Location = System::Drawing::Point(20, 40);
			this->USBType->Name = S"USBType";
			this->USBType->Size = System::Drawing::Size(57, 17);
			this->USBType->TabIndex = 2;
			this->USBType->Text = S"USB";
			this->USBType->CheckedChanged += new System::EventHandler(this, &VISAExampleForm::USBType_CheckedChanged_1);
			// 
			// GPIBType
			// 
			this->GPIBType->Checked = true;
			this->GPIBType->Location = System::Drawing::Point(20, 16);
			this->GPIBType->Name = S"GPIBType";
			this->GPIBType->Size = System::Drawing::Size(64, 17);
			this->GPIBType->TabIndex = 1;
			this->GPIBType->TabStop = true;
			this->GPIBType->Text = S"GPIB";
			this->GPIBType->CheckedChanged += new System::EventHandler(this, &VISAExampleForm::GPIBType_CheckedChanged_1);
			// 
			// lanaddr
			// 
			this->lanaddr->Location = System::Drawing::Point(112, 88);
			this->lanaddr->Name = S"lanaddr";
			this->lanaddr->Size = System::Drawing::Size(128, 20);
			this->lanaddr->TabIndex = 5;
			this->lanaddr->Text = S"TCPIP0::<IP>";
			this->lanaddr->TextChanged += new System::EventHandler(this, &VISAExampleForm::lanaddr_TextChanged_1);
			// 
			// usbaddr
			// 
			this->usbaddr->Location = System::Drawing::Point(112, 64);
			this->usbaddr->Name = S"usbaddr";
			this->usbaddr->Size = System::Drawing::Size(128, 20);
			this->usbaddr->TabIndex = 3;
			this->usbaddr->Text = S"USB0::2391::2567::<sn>";
			this->usbaddr->TextChanged += new System::EventHandler(this, &VISAExampleForm::usbaddr_TextChanged_1);
			// 
			// gpibaddr
			// 
			this->gpibaddr->Location = System::Drawing::Point(112, 40);
			this->gpibaddr->Name = S"gpibaddr";
			this->gpibaddr->Size = System::Drawing::Size(128, 20);
			this->gpibaddr->TabIndex = 0;
			this->gpibaddr->Text = S"GPIB0::22";
			this->gpibaddr->TextChanged += new System::EventHandler(this, &VISAExampleForm::gpibaddr_TextChanged_1);
			// 
			// label4
			// 
			this->label4->Location = System::Drawing::Point(248, 88);
			this->label4->Name = S"label4";
			this->label4->Size = System::Drawing::Size(152, 17);
			this->label4->TabIndex = 88;
			this->label4->Text = S"<IP> = DMM IP Address";
			this->label4->TextAlign = System::Drawing::ContentAlignment::MiddleLeft;
			// 
			// label3
			// 
			this->label3->Location = System::Drawing::Point(248, 64);
			this->label3->Name = S"label3";
			this->label3->Size = System::Drawing::Size(152, 20);
			this->label3->TabIndex = 87;
			this->label3->Text = S"<sn> = DMM Serial Number";
			this->label3->TextAlign = System::Drawing::ContentAlignment::MiddleLeft;
			// 
			// label2
			// 
			this->label2->Location = System::Drawing::Point(248, 40);
			this->label2->Name = S"label2";
			this->label2->Size = System::Drawing::Size(148, 17);
			this->label2->TabIndex = 86;
			this->label2->Text = S"22 = default GPIB address";
			this->label2->TextAlign = System::Drawing::ContentAlignment::MiddleLeft;
			// 
			// listBox1
			// 
			this->listBox1->Location = System::Drawing::Point(36, 120);
			this->listBox1->Name = S"listBox1";
			this->listBox1->Size = System::Drawing::Size(336, 134);
			this->listBox1->TabIndex = 90;
			this->listBox1->TabStop = false;
			// 
			// GetReadings
			// 
			this->GetReadings->Location = System::Drawing::Point(156, 264);
			this->GetReadings->Name = S"GetReadings";
			this->GetReadings->Size = System::Drawing::Size(96, 32);
			this->GetReadings->TabIndex = 6;
			this->GetReadings->Text = S"Get Readings";
			this->GetReadings->Click += new System::EventHandler(this, &VISAExampleForm::GetReadings_Click_1);
			// 
			// ExitProg
			// 
			this->ExitProg->Location = System::Drawing::Point(156, 304);
			this->ExitProg->Name = S"ExitProg";
			this->ExitProg->Size = System::Drawing::Size(96, 32);
			this->ExitProg->TabIndex = 7;
			this->ExitProg->Text = S"Exit";
			this->ExitProg->Click += new System::EventHandler(this, &VISAExampleForm::ExitProg_Click_1);
			// 
			// VISAExampleForm
			// 
			this->AutoScaleBaseSize = System::Drawing::Size(5, 13);
			this->ClientSize = System::Drawing::Size(409, 352);
			this->Controls->Add(this->GetReadings);
			this->Controls->Add(this->ExitProg);
			this->Controls->Add(this->listBox1);
			this->Controls->Add(this->label4);
			this->Controls->Add(this->label3);
			this->Controls->Add(this->label2);
			this->Controls->Add(this->lanaddr);
			this->Controls->Add(this->usbaddr);
			this->Controls->Add(this->gpibaddr);
			this->Controls->Add(this->SelectIOType);
			this->Controls->Add(this->Label1);
			this->Name = S"VISAExampleForm";
			this->StartPosition = System::Windows::Forms::FormStartPosition::CenterScreen;
			this->Text = S"VISA Example";
			this->SelectIOType->ResumeLayout(false);
			this->ResumeLayout(false);
			this->PerformLayout();

		}	


	//////////////////////////////////////////////////////////////////////////////////////
	private: System::Void GetReadings_Click_1(System::Object*  sender, System::EventArgs*  e) 
	// Main function that calls the functions to makes and return measurements
	{

		// Clear the list box
		listBox1->Items->Clear();

		// If no connection is made to the instrument, make the connection
		if (!connected)
			if (!open_port())
				return;

		// Disable buttons
		GetReadings->Enabled = false;
		ExitProg->Enabled = false;

		// Call setup to setup the multimeter
		setup();

		// Call readings to trigger the multimeter and return the readings
		readings();

		GetReadings->Enabled = true;
		ExitProg->Enabled = true;
		ExitProg->Focus();
	}

	//////////////////////////////////////////////////////////////////////////////////////
	System::Void setup()
	// This function performs the instrument setup.
	{
		ViChar	tempval[5];

		// Check for exceptions
		try
		{
			// Reset to turn-on conditions
			errorStatus = viPrintf(vi, "*RST\n");

			// Clear the status registers
			errorStatus = viPrintf(vi, "*CLS\n");

			// Wait for commands to complete
			errorStatus = viQueryf(vi, "*OPC?\n", "%t", tempval);

			// Setup the registers. Only need this if using the registers to determine when
			// the measurements are complete (see "readings" function for more information
			// Enable the "operation Complete" bit (bit 1) of the Standard Event Enable Register
			errorStatus = viPrintf(vi, "*ESE 1\n");

			// Enable the "Standard Event" (bit 6) the Status Byte
			errorStatus = viPrintf(vi, "*SRE 32\n");

			// Wait for commands to be executed
			errorStatus = viQueryf(vi, "*OPC?\n", "%t", tempval);

			// Setup the 34410A, 34411A
			// Configure the instrument to measure DCV on the 10V range; use your own
			// configuration
			errorStatus = viPrintf(vi, "CONFigure:VOLTage:DC 10\n")	;

			// Setup instrument to trigger on a BUS trigger (*TRG) command; use your own source
			errorStatus = viPrintf(vi, "TRIGger:SOURce BUS\n");

			// Set a trigger delay value in seconds (optional); use your own value
			errorStatus = viPrintf(vi, "TRIGger:DELay 1\n");

			// Setup instrument to take 10 readings per trigger; use your own value
			errorStatus = viPrintf(vi, "SAMPle:COUNt 10\n");

			// Place instrument in wait for trigger mode and trigger it
			errorStatus = viPrintf(vi, "INITiate\n");

			// Determine measurement time

			// Get the selected delay value
			errorStatus = viQueryf(vi, "TRIGger:DELay?\n", "%lf", &tottime);

			// Get the number of samples per tirgger
			errorStatus = viQueryf(vi, "SAMPle:COUNt?\n", "%d", &numrdgs);

			// Calculate the total time
			tottime = Math::Round((tottime * ((double)numrdgs)), 1);
				
			// Check for errors
			check_error("setup");

		}
		catch (Exception* ex)
		{
			MessageBox::Show(ex->get_Message(), "Program Error in function: setup", MessageBoxButtons::OK, MessageBoxIcon::Stop);

			// Close the session
			CloseSession();

			// Exit program
			exit(1);
		}
	}

	//////////////////////////////////////////////////////////////////////////////////////
	System::Void readings()
	// This function triggers the instrument, makes the measurements, and returns readings.
	{
		String*		showmsg;
		ViUInt16	stb = 0;
		int			numrdgs = 0;
		char		*rdgset,
					*rdgret;
		float		rdgs[50000]={'\0'};
		int			i = 0;

		// Check for exceptions
		try
		{
			listBox1->Items->Add(S"Making measurements; please wait.");
			showmsg = tottime.ToString();
			showmsg = showmsg->Concat(S"Measurement time is about: ", showmsg, S" seconds.");
			listBox1->Items->Add(showmsg);
			listBox1->Refresh();

			// Trigger instrument
			errorStatus = viPrintf(vi, "*TRG\n");

			// If using GPIB, there are 4 ways to determine when all measurements are made, USB
			// and LAN can use 3 ways, as follows:
			// 1) Read bit 7 of the Status Byte to determine when operation completes. This
			//    can be done using the "*STB?" command or use the Dmm.IO.ReadSTB method
			// 2) Read bit 1 of the Standard Event Register using "*ESR?"
			// 3) Read the reading memory using "DATA:POINts?" to determine the number of
			//    readings stored in memory; these can be removed at any time even during
			//    measurements
			// 4) Set a timeout sufficient for the measurements to be made and then execute
			//    "*OPC?" to determine when the operation completes
			// If using LAN or USB, only #2, #3, or #4 can be used;
			// This example uses #1 for GPIB or #2 for LAN and USB
			
			// Set "Operation Complete" bit (bit 1) in Standard Event Register
			errorStatus = viPrintf(vi, "*OPC\n");
			
			if (GPIBType->Checked)
			{
				// Read the status byte until bit 7 is set, indicating that measurements are complete
				do
				{
					errorStatus = viReadSTB(vi, &stb);
				}
				while ((stb & 64) != 64);
			}
			else
			{
				do
				{
					// Read bit 1 of the Standard Event Register until bit 1 is set
					errorStatus = viQueryf(vi, "*ESR?\n", "%d", &stb);
				}
				while (stb!= 1);
			}

			// There are 3 different ways to return readings, as follows:
			// R? = This returns all readings in memory and clears the memory. Readings are
			//   returned as definite-length block data as ASCII values seperated by commas, 
			//   or REAL: 64/REAL: 32 values 
			// DATA:REMove? = returns a selected number of readings and clears memory. Readings 
			//   are returned as ASCII values seperated by commas, or REAL: 64/REAL: 32 as 
			//   definite-length block data
			// FETCh? = returns all readings in ASCII format without removing the readings from
			//   memory.
			// In this example, all readings are returned as arbitrary block data as REAL: 32 values

			// Get number of readings in memory
			errorStatus = viQueryf(vi, "DATA:POINTs?\n", "%d", &numrdgs);

			// Select the returned reading type (ASCII, REAL: 32, or REAL: 64)
			errorStatus = viPrintf(vi, "FORMat:DATA REAL,32\n");

			// If returning the readings using definite-length block data, sent the seelct the 
			// correct byte order. Most computers use the normal order (Most Significant Bit
			// first), but some may use the reverse order (Least Significant Bit first).
			errorStatus = viPrintf(vi, "FORMat:BORDer NORMal\n");

			// Setup the rdgset and rdgret variables for the viQueryf function to return
			// the reading data. The viQueryf function using the variables looks like this:
			// viQueryf(vi, "DATA:REMove? 10", "%10zb", rdgs);
			showmsg = showmsg->Concat(S"DATA:REMove? ", (numrdgs.ToString()), S"\n");
			rdgset = (char*)Marshal::StringToHGlobalAnsi(showmsg).ToPointer();
			showmsg = showmsg->Concat(S"%", (numrdgs.ToString()), S"zb"); 
			rdgret = (char*)Marshal::StringToHGlobalAnsi(showmsg).ToPointer();

			// Use the formatted viQueryf function to return the readings; this function used
			// used since the returned readings are not ASCII values, but definite-length
			// arbitrary block data.
			errorStatus = viQueryf(vi, rdgset, rdgret, rdgs);

			// Since the viQueryf function only returns the block data, the ASCII character 10 value
			// returned by the instrument needs to be read to prevent errors when sending 
			// a new query
			errorStatus = viScanf(vi, "%i", &i);

			// List readings
			listBox1->Items->Clear();
			showmsg = S"Rdng# \tValue";
			listBox1->Items->Add(showmsg);

			for (i=0;i<numrdgs;i++)
			{
				showmsg = showmsg->Concat((i + 1).ToString(), S"\t", (rdgs[i]).ToString());
				listBox1->Items->Add(showmsg);
			}

			// Check for errors
			check_error("readings");

			listBox1->Items->Add(S"");
			listBox1->Items->Add(S"Select the I/O Type, enter instrument address, if needed;");
			listBox1->Items->Add(S"then click on \"Get Readings\". Measurement will take some time");

		}
		catch (Exception* ex)
		{
			MessageBox::Show(ex->get_Message(), "Program Error in function: readings", MessageBoxButtons::OK, MessageBoxIcon::Stop);

			// Close the session
			CloseSession();

			// Exit program
			exit(1);
		}
	}

	//////////////////////////////////////////////////////////////////////////////////////
	System::Boolean open_port()
	// This function makes connections to the instrument and checks for the correct instrument
	{
		String* addr;
		char *addrval;
		ViChar	msg[256];
		String*	retval;

		// Check for exceptions			
		try
		{
			// Close session just in case it is open
			CloseSession();

			// Get address
			if (GPIBType->Checked)	// GPIB Selected
				addr = String::Concat((gpibaddr->Text), S"::INSTR");
			if (USBType->Checked)	// USB Seelcted
				addr = String::Concat((usbaddr->Text), S"::0::INSTR");
			if (LANType->Checked)	// LAN Selected
				addr = String::Concat((lanaddr->Text), S"::inst0::INSTR");

			// Change string array to char
			addrval = (char*) Marshal::StringToHGlobalAnsi(addr).ToPointer();

			// Open the VISA session
			errorStatus = viOpenDefaultRM(&videfaultRM);

			// Open communication to the instrument
			errorStatus = viOpen(videfaultRM, addrval, VI_TRUE, VI_TRUE, &vi);
		
			// If an error occurs, give a message
			if (errorStatus < VI_SUCCESS)
			{
				MessageBox::Show("Unable to Open port; check address!", "34410A_34411A Example Program", MessageBoxButtons::OK, MessageBoxIcon::Stop);
			
				// Set to initial address value
				if (GPIBType->Checked)	// GPIB Selected
					gpibaddr->Text = S"GPIB0::22";
				if (USBType->Checked)	// USB Seelcted
					usbaddr->Text = S"USB0::2391::2567::<sn>";
				if (LANType->Checked)	// LAN Selected
					lanaddr->Text = S"TCPIP0::<IP>";

				// Set GPIB to initial value
				gpibaddr->Text = "GPIB0::22";
			
				// Close the session
				CloseSession();

				return false;
			}
			// Check and make sure the correct instrument is addressed
			errorStatus = viQueryf(vi,"*IDN?\n","%t", msg);

			// Store into String array
			retval = msg;
		
			// Check if a string was stored into msg; store a bogus string if length is < 5
			if ((retval->Length) < 5)
				retval = "XXXXX\n";

			// Remove the linefeed character
			retval=retval->Remove((retval->Length) -1,1);
			if (retval->IndexOf("34410A") < 0)
				if (retval->IndexOf("34411A") < 0)
				{
					MessageBox::Show("Incorrect instrument selected; use the correct address.", "34410A_34411A Example Program", MessageBoxButtons::OK, MessageBoxIcon::Stop);

					// Set to initial address value
					if (GPIBType->Checked)	// GPIB Selected
						gpibaddr->Text = S"GPIB0::22";
					if (USBType->Checked)	// USB Seelcted
						usbaddr->Text = S"USB0::2391::1287::<sn>";
					if (LANType->Checked)	// LAN Selected
						lanaddr->Text = S"TCPIP0::<IP>";

					// Close the session
					CloseSession();
					return false;
				}

			// Add ID string to list box
			listBox1->Items->Add(retval);
			listBox1->Items->Add(S"");
	
			connected=true;	
			return true;
		}
		catch (Exception* ex)
		{
			MessageBox::Show(ex->get_Message(), "Program Error in function: open_port", MessageBoxButtons::OK, MessageBoxIcon::Stop);

			// Close the session
			CloseSession();

			// Exit program
			exit(1);
		}
		return true;
	}

	//////////////////////////////////////////////////////////////////////////////////////
	System::Void check_error(char *msg)
	// Checks for syntax and other errors.
	{
		ViChar	err_msg[256];
		ViInt16	err_num;
		String*	tempval;
		String* tempmsg;

		// Check for first errors
		errorStatus = viQueryf(vi, "SYSTem:ERRor?\n", "%d,%t", &err_num, err_msg);

		// If error is found, show the error(s)
		if (err_num)
		{       
			while(err_num)
			{
				// Store error message into String array
				tempval = err_msg;
				tempmsg = msg;
				
				// Display error in a message box
				tempval = tempval->Concat(S"Error: ", (err_num).ToString(), S", ", tempval, S"in function: ", tempmsg);
				MessageBox::Show(tempval, "34410A_34411A Example Program", MessageBoxButtons::OK, MessageBoxIcon::Stop);
				
				errorStatus = viQueryf(vi, "SYSTem:ERRor?\n", "%d,%t", &err_num, err_msg);
			}
			// Send a device clear
			viPrintf(vi, "*CLS\n");
	    
			// Close the session
			CloseSession();

			// Exit program
			Environment::Exit(1);
		}
	}

	//////////////////////////////////////////////////////////////////////////////////////
	System::Void CloseSession()
	// This functions closes a session and set the conencted mode false.
	{
		viClose(vi);
		viClose(videfaultRM);
		connected = false;
	}

private: System::Void ExitProg_Click_1(System::Object*  sender, System::EventArgs*  e) 
		 {
			 if (connected)
				 CloseSession();
			 Environment::Exit(0);
		 }
private: System::Void GPIBType_CheckedChanged_1(System::Object*  sender, System::EventArgs*  e) 
		 {
			 if (connected)
				 CloseSession();
			connected = false;
		 }
private: System::Void USBType_CheckedChanged_1(System::Object*  sender, System::EventArgs*  e) 
		 {
			 if (connected)
                 CloseSession();
			connected = false;
		 }
private: System::Void LANType_CheckedChanged_1(System::Object*  sender, System::EventArgs*  e) 
		 {
			 if (connected)
                 CloseSession();
			connected = false;
		 }
private: System::Void gpibaddr_TextChanged_1(System::Object*  sender, System::EventArgs*  e) 
		 {
			 if (connected)
                 CloseSession();
			connected = false;
		 }
private: System::Void usbaddr_TextChanged_1(System::Object*  sender, System::EventArgs*  e) 
		 {
			 if (connected)
                 CloseSession();
			connected = false;
		 }
private: System::Void lanaddr_TextChanged_1(System::Object*  sender, System::EventArgs*  e) 
		 {
			 if (connected)
                 CloseSession();
			 connected = false;
		 }
};
}
